# Updated 2012.07.20 to interpret quoted indices with spaces (by using csv.reader)
# Updated 20120814 to solve both GMPL and AMPL models
# This allows us to parse AMPL model files to distinguish between 
# singletons that can be params and single-member sets 
#
# v1.41, 2013.06.03; Closed the results file
# 20130301 Changed code to write data items using a _display command
# 20140407 Added (missing) support for reading data written using _display, and fixed bug stopping display always being converted to _display
# 20140620 Removed any :g15 formatting as this is locale specific, and so fails in non-English countries

#language = "GMPL"
language = "AMPL"
defaultSolver = "cbc"

import re
import os
import sys
import clr
import shutil
import shlex
import csv
clr.AddReference('System.Windows.Forms')
from System.Windows.Forms import Form, Label, ToolStripMenuItem, MessageBox, MessageBoxButtons, DialogResult

def BuildNameSub(f):
	#
	#	Handle index substitutions of the form
	#
	#	PC_Flow [*,*]
	#	# $4 = 'Customer 4'
	#	# $6 = 'Customer 6'
	#	:         'Customer 1' ...
	#	'Plant 1'     107.9 ...
	#
	subDict={}
	lineCount = 0;
	while True:
		line = f.readline();
		if not line:
			raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an unexpected end of file occurred. (Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")") #columns not read in		cols = csv.reader([line], delimiter=' ', skipinitialspace=True, quotechar="'").next()
		lineCount = lineCount + 1
		line = line.strip()
		if line[0] != '#': break
		items = shlex.split(line)
		subDict[items[1]] = items[3]	
	return line, subDict, lineCount

# Converts singletons (double and strings) to strings (outputting floats containing integers with no decimal point)
def fmtSngl(s):
   # Ensure integers such as 1.0 (showing in Excel as 1, but stored as floats) are output as integers
   # Integers are much more natural for indexing etc than 1.0, 2.0, 3.0 that happens otherwise
   if type(s) is float:
      if s.is_integer():
         return repr(int(s))
   return repr(s)

# Converts tuples to strings (with no surrounding brackets)
def fmtTuple(s):
   r = fmtSngl(s[0])
   for i in s[1:]:
      r=r+", "+fmtSngl(i)
   return r

# Determines whether single or tuple
def fmt(s):
   if type(s) is tuple:
      #f.write('['+','.join([repr(t) for t in s])+']')
      return fmtTuple(s)
   else:
      return fmtSngl(s)

# Return s as a string suitable for indexing, which means tuples are listed items in []
def WriteIndex(f,s):
   if type(s) is tuple:
      #f.write('['+','.join([repr(t) for t in s])+']')
      f.write('['+fmt(s)+']')
   else:
      f.write(fmt(s))

# Return s as a string suitable for an item in a set, which means tuples are listed items in ()
def WriteItem(f,s):
   if type(s) is tuple:
      #f.write( '('+','.join([repr(t) for t in s])+')' )
      f.write( "("+fmt(s)+")" )
   else:
      f.write(fmt(s))

def WriteParam(f,name,value):
   f.write("param "+name+" := "+fmt(value)+";\n\n")

# set NUTR := A B1 B2 C ;
def WriteSet(f,name,value):
   f.write("set "+name+" := ")
   j = 0
   for i in value:
      if j>20:
         j=0
         f.write("\n")
      f.write(" ")
      WriteItem(f,i)
      j = j+1
   f.write(";\n\n")
  
# set NUTR := A ;
def WriteSingletonSet(f,name,value):
   f.write("set "+name+" := ")
   j = 0
   WriteItem(f,value)
   f.write(";\n\n")
 
def WriteIndexedSet(f,name,value):
   for (index,val) in value.iteritems():
      f.write("set ")
      f.write(name)
      f.write("[")
      f.write(fmt(index))
      f.write("] := ")
      if val != None and val != ():   # Write empty rows (which end up as an empty tuple, hence "()" in code) as empty sets (by writing nothing)
         f.write(fmt(val)) # We write values separated by commas...
      f.write(";\n")
   f.write("\n")
 
 
# param demand := 
#   2 6 
#   3 7 
#   4 4 
#   7 8; 
# param sc  default 99.99 (tr) :

def WriteIndexedParam(f,name,value):
   if int(value.BadIndexOption) == 2: # 2 = ReturnUserValue; TODO - refer to the enum correctly
      f.write("param "+name+" default ");
      WriteItem(f,value.BadIndexValue);
      f.write(" := ")
   else:
      f.write("param "+name+" := ")
   for (index,val) in value.iteritems():
      if val != None:   # Skip empty cells (which can still get a value from a 'default' option above; as of v0.5, there should not be any
         f.write("\n  ")
         WriteIndex(f,index)
         f.write(" ")
         WriteItem(f,val)
   f.write("\n;\n\n")
   
# Finds sets, params, vars, and solver in the model
def ScanModelFileForDataItems(modelFileName):
   solveCommand=False
   dataCommand=False
   displayCommand = False
   # widthCommand = False
   # displayWidth = 0
   solver=None;
   Variables=list();
   Sets= list();
   Params = list();
   lineCount = 0
   f = open(SolverStudio.ModelFileName,"r")
   for line in f:
      line = line.strip() # Remove white space, including the newline
      # Replace anything that can follow the name with a 'space' so the string splits correctly.
      l = line.replace("="," ").replace(";"," ").replace("<"," ").replace(">"," ").replace("{"," ").replace(":"," ").replace(","," ")
      lineCount += 1
      tokens = l.split()
      if len(tokens)==0: continue
      if tokens[0].lower()=="param":
         if len(tokens)<2:
            f.close()
            raise Exception("Unable to find the name for the "+tokens[0]+" defined by '"+line+"' on line "+str(lineCount)+" of the AMPL model.")
         if not tokens[1] in Params:
            # print "##  param ",tokens[1]
            Params.append(tokens[1])   
      elif tokens[0].lower()=="set":
         if len(tokens)<2:
            f.close()
            raise Exception("Unable to find the name for the "+tokens[0]+" defined by '"+line+"' on line "+str(lineCount)+" of the AMPL model.")
         if not tokens[1] in Sets:
            # print "##  set ",tokens[1]
            Sets.append(tokens[1])
      elif tokens[0].lower()=="solve":
            solveCommand=True
      elif tokens[0].lower()=="data":
            dataCommand=True
      elif tokens[0].lower()=="display":
            displayCommand=True
      elif tokens[0].lower() == "var":
            Variables.append(tokens[1])
      elif tokens[0].lower() == "option":
            if tokens[1].lower() == "solver":
                solver = tokens[2]
            #elif tokens[1].lower() == "display_width":
            #	displayWidth = tokens[2]
            #	widthCommand = True
   f.close()
   #return Sets, Params,solveCommand,dataCommand,displayCommand,widthCommand,displayWidth,Variables,solver
   return Sets, Params,solveCommand,dataCommand,displayCommand,Variables,solver

def FixBadCommands(model,solveCommand,dataCommand,displayCommand,Variables):
	if not displayCommand:
		if solveCommand and not dataCommand:
			r= re.compile(r'^\s*solve\s*;',re.IGNORECASE | re.MULTILINE)
			model = r.sub("data SheetData.dat;  #SolverStudio: Line added",model)
			model = model + "\n\nsolve;\n\n"
		if not solveCommand and dataCommand:
			model = model + "\nsolve; #SolverStudio: Line added\n\n" 
		if not solveCommand and not dataCommand:
			model = model + "\n\ndata SheetData.dat; #SolverStudio: Line added" + "\n\nsolve; #SolverStudio: Line added\n\n"
		for i in Variables:
			model = model + "_display " + i + " > Sheet; #SolverStudio: Line added\n\n"
	else:
		if solveCommand and not dataCommand:
			r= re.compile(r'^\s*solve\s*;',re.IGNORECASE | re.MULTILINE)
			model = r.sub("data SheetData.dat; #SolverStudio: Line added\n\nsolve;",model)
		if not solveCommand and dataCommand:
			r=re.compile(r'^\s*display\s',re.IGNORECASE| re.MULTILINE)
			model = r.sub("solve; #SolverStudio: Line added\n\ndisplay ",model,1)
		if not solveCommand and not dataCommand:
			r=re.compile(r'^\s*display\s',re.IGNORECASE| re.MULTILINE)
			model = r.sub("\ndata SheetData.dat; #SolverStudio: Line added\n\nsolve; #SolverStudio: Line added\n\ndisplay ",model,1)
		# Convert all the display XXX > Sheet; to _display XXX > Sheet; 
		# _display is the safest most-computer-friendly way to output values, and the one we can read back most reliably
		# This also works around problems with long names for indices which cause unusual wrapping (to do with the display width option; see the AMPL ReadMe.txt file)
		# 20150527: We also now ensure "Sheet" always occurs with a capital S; otherwise previous output is deleted by AMPL thinking sheet and Sheet are different files
		r=re.compile(r'(^\s*)display(\s[a-zA-Z._]+\s*>\s*)Sheet(\s*;)',re.IGNORECASE| re.MULTILINE)
		model = r.sub(r"\1_display\2Sheet\3 #SolverStudio: Line modified",model)
	return model

def FixBadSolver(model,solver):
	newSolver = None
	continueOK = True
	try:
		path = SolverStudio.GetAMPLPath(preferSolverStudioAMPL=SolverStudio.GetRegistrySetting("AMPL","PreferSolverStudioAMPL",1)==1, mustExist=True)
	except:
		raise Exception("AMPL is not installed. Please install the student version using the AMPL menu in SolverStudio or purchase the commercial version.")
	if solver==None:
		solverExists = False
		solver = defaultSolver
		r= re.compile(r'^\s*solve\s*;',re.IGNORECASE | re.MULTILINE)
		m = re.search(r,model)
		if m:
			model = r.sub("\noption solver "+defaultSolver+"; # SolverStudio: Line added\nsolve;",model)
		else:
			model = model + "\noption solver "+defaultSolver+"; # SolverStudio: Line added\nsolve; # SolverStudio: Line added\n"
	return model, continueOK
	#
	# Look for the solver in the ampl directory, or the solvers folders, or in the path
	solverPath = SolverStudio.FindPathForSolver(solver, path, mustExist=False);
	solverExists = solverPath != None
	oldSolver=solver
	if not solverExists:
		if solver == "cplex": 
			newSolver = "cplexamp"
		elif solver == "cplexamp": 
			newSolver = "cplex"
		else: 
			result = MessageBox.Show("The solver " + repr(solver) + " was not found.\nUse default AMPL solver instead?","SolverStudio",MessageBoxButtons.YesNoCancel)
		if result == DialogResult.Yes: 
			newSolver = "Default"
		elif result == DialogResult.Cancel: 
			continueOK=False
			return model,continueOK
		if newSolver != "" and newSolver != None:
			r = re.compile(solver)
			path = r.sub(newSolver,path)
			solverExists = os.path.isfile(path)
			solver = newSolver
			r=re.compile(r"option\s*solver")
			if newSolver == "Default":
				model=r.sub("#SolverStudio: Let AMPL choose default solver",model)
			elif solver == "cplex" or solver == "cplexamp": 
				model=r.sub('option solver '+ solver +'; ##SolverStudio: Line modified',model)
			print "## Fixing choice of solver"
	return model,continueOK

def DoSmallFixes(solveCommand,dataCommand,displayCommand,Variables,solver):
   with open(SolverStudio.ModelFileName,"r") as input_file:
      model=input_file.read()
   #Fix missing data,solve,display commands
   print "## Fixing model commands"
   model=FixBadCommands(model,solveCommand,dataCommand,displayCommand,Variables)
   # if solver:
   model,continueOK=FixBadSolver(model,solver)

   with open(SolverStudio.ModelFileName,"w") as dest_file:
      dest_file.write(model)
   return continueOK

def Read_displayItem(line, g, items, lineCount, itemsLoaded, itemsChanged, standardErrorHelp):
	# Read something like:
   # _display 2 1 50  <- NumIndices, NumValues, NumRows
   # flow
   # A,1,0
   # A,2,0
   # ...
   # J,4,0
   # J,5,0
	
	numIndices = int(items[1])
	numValues = int(items[2])
	numRows = int(items[3])
	name = g.readline()
	lineCount += 1
	name = name.strip()  # remove line feed at end
	escapedName = SolverStudio.EscapedDataItemName(name) # The Python name used internally
	if name=="":
			raise Exception("An unexpected entry has occurred when writing the new values to the sheet; the data item name is missing in the temporary file.\n\n"
			+standardErrorHelp+"\n\n(The error occurred in the string '"+repr(line)+"' on line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)\n\n")
	try:      
		dataItem = SolverStudio.DataItems[escapedName]
	except:
		raise Exception("A value was written to the sheet for data item '"+repr(name)+"', but this data item has not been defined in your spreadsheet.\n\n"
		+standardErrorHelp+"\n\n(The error occurred in the string "+repr(line)+" on line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)")
	changed = False
	for i in range(numRows):
		line = g.readline()
		lineCount += 1
		items = csv.reader([line], delimiter=',', skipinitialspace=True, quotechar="'").next()
		if len(items) != numIndices+numValues:
			raise Exception("An unexpected number of _display arguments was found when reading the new value for data item '"+repr(name)+"'.\n\n"
			+standardErrorHelp+"\n\n(The error occurred in the string "+repr(line)+" on line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)")
		if numIndices==0:
			value = items[-1]
			try:
				value = float(items[-1])
			except ValueError:
				pass # We don't throw an error because this may be  string
				#raise Exception("When processing the value written to the sheet for data item '"+repr(name)+"', an error occurred when converting the item '"+repr(items[-1])+"' to a number.\n\n"+standardErrorHelp+"\n\n(The error occurred in string '"+repr(line)+"' in line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
			try:
				if dataItem!=value:
					# This fails inside a function; needs globals() instead
					# vars()[name]=value # Update the python variable of this name (without creating a new variable, which would not be seen by SolverStudio               
					globals()[escapedName]=value # Update the python variable of this name (without creating a new variable, which would not be seen by SolverStudio)
					changed = True
			except:
				raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred when assigning the value "+repr(items[-1])+" to "+repr(name)+".\n\n"+standardErrorHelp+"\n\n(The error occurred in string '"+repr(line)+"' in line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)")
		else:
				value = items[-1]
				try:
					value = float(items[-1])
				except ValueError:
					pass	# We allow string values as well
					# raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred in line "+repr(line)+" when converting the last item '"+repr(items[-1])+"' to a number.\n\n"
					#+standardErrorHelp+"\n\n(Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
				index = items[0:-1]
				# Convert any numeric values into floats (leaving strings such as letters unchanged)
				for j in range(len(index)):
					try:
						index[j]=float(index[j])
					except ValueError:
						pass
				#print tuple(index),value
				#try:
				if len(index)==1:
					index2 = index[0]  # The index is a single value, not a tuple
				else:
					index2 = tuple(index)# The index is a tuple of two or more items
				if dataItem[index2]!=value:
					dataItem[index2]=value
					changed = True
	if changed:
		itemsLoaded = itemsLoaded+name+"* "
		itemsChanged = True
	else:
		itemsLoaded = itemsLoaded+name+" "
	return items, lineCount, itemsLoaded, itemsChanged

#######################################################################

# The main run code must be in a subroutine to avoid clashes with SolverStudio globals (which can cause COM exceptions in, eg, "for (name,value) in SolverStudio.SingletonDataItems.iteritems()" if value exists as a data item)
def DoRun():
	# duplicate the model file so it is available as a xxx.mod file ready to be loaded into an AMPL shell
	shutil.copy(SolverStudio.ModelFileName, os.path.splitext(SolverStudio.ModelFileName)[0] + ".mod");

	# Create a 'data' file, that contains the appropriate data
	dataFilename = "SheetData.dat"
	g = open(dataFilename,"w")
	g.write("# "+language+" data file created by SolverStudio\n\n")
	g.write("data;\n\n")

	print "## Scanning model for sets and parameters"
	Sets, Params,solveCommand,dataCommand,displayCommand,Variables,solver = ScanModelFileForDataItems(SolverStudio.ModelFileName)
	#print "Sets=",Sets
	#print "Params=",Params

	continueOK=True
	succeeded = False
	if SolverStudio.ActiveModelSettings.GetValueOrDefault("FixMinorErrors","True"):
		print "## Attempting to fix minor errors"
		continueOK = DoSmallFixes(solveCommand,dataCommand,displayCommand,Variables,solver)
	if continueOK:
		print "## Building input file for %s sets and %s parameters"%(len(Sets),len(Params))

		print "## Writing simple parameters...",
		# Create constants
		count = 0
		for (escapedName,value) in SolverStudio.SingletonDataItems.iteritems():
			name = SolverStudio.UnEscapedDataItemName(escapedName)
			if name in Params:
				count += 1
				WriteParam(g, name,value)
		print count,"items written."
				
		print "## Writing sets...", #  Sets may be Python singletons or lists
		# Create sets
		# print Sets, SolverStudio.DictionaryDataItems
		count = 0
		for (escapedName,value) in SolverStudio.SingletonDataItems.iteritems():
			name = SolverStudio.UnEscapedDataItemName(escapedName)
			if name in Sets:
				count += 1
				WriteSingletonSet(g, name,value)
		for (escapedName,value) in SolverStudio.ListDataItems.iteritems():
			name = SolverStudio.UnEscapedDataItemName(escapedName)
			if name in Sets:
				count += 1
				WriteSet(g, name,value)
		for (escapedName,value) in SolverStudio.DictionaryDataItems.iteritems():
			name = SolverStudio.UnEscapedDataItemName(escapedName)
			if name in Sets:
				count += 1
				WriteIndexedSet(g, name,value)
		print count,"items written."
			
		print "## Writing indexed parameters...",
		# Create indexed params
		count = 0
		for (escapedName,value) in SolverStudio.DictionaryDataItems.iteritems():
			name = SolverStudio.UnEscapedDataItemName(escapedName)
			if name in Params:
				count += 1
				WriteIndexedParam(g, name,value)
		print count,"items written."

		g.write("end;\n") # Required for GMPL (and optional in AMPL?)

		g.close()

		# Ensure the file "Sheet" that AMPL/GMPL writes its results into is empty
		# (We don't just delete it, as that require import.os to use os.remove)
		resultsfilename = "Sheet"
		g = open(resultsfilename,"w")
		g.close()

		preferSolverStudioAMPL=SolverStudio.GetRegistrySetting("AMPL","PreferSolverStudioAMPL",1)==1
		if language=="AMPL":
			try:
				# Get AMPL exe
				exeFile =SolverStudio.GetAMPLPath(preferSolverStudioAMPL=preferSolverStudioAMPL, mustExist=True)
			except:
				raise Exception("AMPL does not appear to be installed. Please install the student version using the AMPL menu in SolverStudio or purchase the commercial version.")
		else:
			exeFile = SolverStudio.GetGLPSolPath()

		if language=="AMPL":
			# We start the AMPL license mgr if it exists in the same directory as the exe file, and it is not currently running
			# We do this as if we leave this to AMPL, the AMPL process we start never completes.
			# See http://www.gurobi.com/documentation/5.0/ampl-gurobi-guide/ampl_lic
			AMPLLicenseMgrPath = SolverStudio.GetAssociatedAMPLLicenseMgr(exeFile)
			RunningAMPLLicenseMgrPath = SolverStudio.GetRunningAMPLLicenseMgrPath()
			if (AMPLLicenseMgrPath!=None) and (RunningAMPLLicenseMgrPath==None):
				print "## Starting AMPL License Manager: "+AMPLLicenseMgrPath
				succeeded, error = SolverStudio.StartAMPLLicenseMgr(AMPLLicenseMgrPath,"start")
				if succeeded:
					print "## Successfully started "+AMPLLicenseMgrPath
				else:
					print "## Unable to start "+AMPLLicenseMgrPath+":\n"+error
					print "## WARNING: You may need to manually cancel your AMPL run if a license is needed but not found."

		print "## Running "+language+"..."
		print "## "+language+".exe file:",exeFile
		print "## "+language+" model file:",SolverStudio.ModelFileName
		print "## "+language+" data file:",dataFilename+"\n"

		if language=="AMPL":
			try:
				# Run AMPL, ensuring we add the AMPL directory to the path which allows the student version to find the solver such as CPLEX in this folder
				# We also add the Solvers path (being either the 32 or 64 bit folder)
				# We add these paths at the start of %PATH% if we are running the SolverStudio AMPL (to prevent any solvers installed by the user being 
				# seen first if these clash with the AMPL solvers), and at the end otherwise. This ensures no licences are needed if preferSolverStudioAMPL=true
				# We specify killChildProcesses to kill any processes started by AMPL such as sw.exe that is opened when Excel starts the license mgr, leaving these stops Excel quitting 
				# We specify useLoopingWaitForExit as this prevents SolverStudio hanging if AMPL tries to launch the license mgr, and fails (which happens after we have tried a first time and failed). This has a slight risk of missing some text output.
				exitCode = SolverStudio.RunExecutable(exeFile,'"'+SolverStudio.ModelFileName+'"', addExeDirectoryToPath=True, addSolverPath=True, addAtStartOfPath=preferSolverStudioAMPL, killChildProcesses=True, useLoopingWaitForExit=True)
			except:
				succeeded = False
				raise
		else:
			exitCode = SolverStudio.RunExecutable(exeFile,"--model \""+SolverStudio.ModelFileName+"\""+" --data \""+dataFilename+"\"")

		print

	if exitCode==0:
		print "## "+language+" run completed. Opening results file..."
		try:
			g = open(resultsfilename,"r")

			# Read a file such as AMPL might produce:
			#flow :=
			#A 1   0
			#B 5   0
			#;
			#age = 22
			# Amount [*] :=     <<<<< Note the [*] which we work around
			#    BEEF  60
			# CHICKEN   0
			#     GEL  40
			#  MUTTON   0
			#    RICE   0
			#   WHEAT   0
			# ;
			# TODO: We have switched to csv.reader; we can improve the code as a result
			lineCount = 0
			itemsLoaded = ""
			itemsChanged = False
			# standardErrorHelp = "Please check that your model has one or more 'display' commands each displaying one item; e.g.\n\ndisplay rosteredOn > Sheet;\ndisplay rosteredOff > Sheet;\n"
			standardErrorHelp = "" # 2014.10.01 Not sure that the standard error help makes any sense now!
			while True :
				line = g.readline()
				# Break if end of file
				if not line: break
				lineCount += 1
				# Remove newline '\n'
				line = line.strip()
				if line=="": continue # skip blank lines
				# Parse the line into its space delimited components, recognising quoted items containing spaces
				items = shlex.split(line)
				# Skip blank lines
				if len(items)==0: continue
				# Check for a _display item, and handle it with a specific handler that consumes all the associated lines
				if items[0] == '_display':
					items, lineCount, itemsLoaded, itemsChanged = Read_displayItem(line, g, items, lineCount, itemsLoaded, itemsChanged, standardErrorHelp)
					continue # Start the loop again for a new item
				name = items[0]
				# Handle empty items which are written as, eg: nbr; #empty
				if name[-1]==";": name = name[0:-1] # strip the trailing ;
				escapedName = SolverStudio.EscapedDataItemName(name) # The Python name used internally with, eg A.B escaped as A_00_B
				# print "'"+name+"'="
				if name=="" or name==":":
					raise Exception("An unexpected entry has occurred when writing the new values to the sheet; the data item name is missing in the temporary file.\n\n"
					+standardErrorHelp+"\n\n(The error occurred in the string '"+repr(line)+"' on line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)\n\n")
				try:      
					dataItem = SolverStudio.DataItems[escapedName]
				except:
					raise Exception("A value was written to the sheet for data item '"+repr(name)+"', but this data item does not exist.\n\n"
				 +standardErrorHelp+"\n\n(The error occurred in the string "+repr(line)+" on line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)")
				changed = False
				if len(items)==2 and items[0][-1]==";" and items[1]=="#empty":
					# Handle empty items which are written (with no preceding _display) as, eg: nbr; #empty
					pass # We simply skip it. TODO: But, we should really clear it to be an empty dict, or an empty scalar, assuming AMPL can actually modify a value to become empty?
				elif len(items)==3 and items[1]=="=":
					# Read a line like: Age = 22 or 'Boys Age' = 22
					try:
						value = float(items[-1])
					except:
						raise Exception("When processing the value written to the sheet for data item '"+repr(name)+"', an error occurred when converting the last item '"+repr(items[-1])+"' to a number.\n\n"+standardErrorHelp+"\n\n(The error occurred in string '"+repr(line)+"' in line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
					try:
						if dataItem!=value:
							globals()[escapedName]=value # Update the python variable of this name (without creating a new variable, which would not be seen by SolverStudio)
							# vars() and globals() both work ok for this as we are not inside a method
							changed = True
					except:
						raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred when assigning the value "+repr(items[-1])+" to "+repr(name)+".\n\n"+standardErrorHelp+"\n\n(The error occurred in string '"+repr(line)+"' in line "+repr(lineCount)+" of file "+repr(resultsfilename)+".)")
				# For GMPL we do not check for only a :=, but also allow =
				elif items[-1] != ":=" and items[-1] != "=":
					#raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred in line "+repr(line)+"; '"+repr(items[-1])+"' was expected to be ':='. (Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")\n\n"+standardErrorHelp)
					# Read lines such as: 
					#flow :=    **OR, if we have one index**   flow [*] :=
					#'A B' 1   0
					#B 5   0
					#;
					# We also handle a file like the following, which happens with large data items
					# unless the following is used to force lists (not tables): option display_1col 99999999;
					#       rosteredOn [*,*] (tr)
					#       :   H1  H2  H3  H4  N1  N2  N3  N4  R1  R2  R3  R4  R5    :=
					#       0    1   0   0   0   1   0   0   0   0   0   1   0   0
					#       1    1   0   0   0   1   0   0   0   0   0   1   0   0
					#
					#       :  R6  R7 :=
					#       0   1   1 
					#       1   7   8
					#       ;
					# We also handle modifications of this in which some indices are specified, eg rosteredOn [*,'Tuesday',*]
					#Check transpose
					if len(items)==3 and items[2]=="(tr)":
						transpose=True
					else:
						transpose=False
					NotFinished=True
					while NotFinished:
						# Work thru each header, eg ":   H1  H2  H3  H4  N1  N2  N3  N4  R1  R2  R3  R4  R5    :="  ...
						place=0
						numIndex=0
						stars=[]
						indices=[]
						lastComma=0
						currentIndex=1
						try:
							for i in items[-1-transpose][1:]:
								if i=="*":
									stars.append(numIndex)
									place+=1
								elif i=="," or i=="]":
									indices.append(items[-1-transpose][lastComma+1:currentIndex])
									numIndex += 1
									lastComma=currentIndex
								currentIndex += 1
							numIndex+=1
						except:
							raise Exception("An unexpected error occurred when reading the data.\n\nThis could be due to long indices, please try including the line:\n\noption display_width\n\nFollowed by a length larger than your longest index.")
						#line=g.readline()
						line, sDict, count = BuildNameSub(g)#	Modified to parse index substitutions eg in [*,'Tuesday',*]
						lineCount += count
						if not line:
							raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an unexpected end of file occurred. (Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")") #columns not read in
						cols = shlex.split(line)
						cols = cols[1:-1]
						#
						#	cols are the columns for the data structure
						#	you can recognise them with the ": [...] :=" format
						#	:    20  21  22  23  24  25    :=
						#
						for i in range(0,len(cols)):#	Index Substitutions
							if cols[i] in sDict:
								cols[i] = sDict[cols[i]]		
						while True:
							line = g.readline()
							lineCount += 1
							if not line: 
								raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an unexpected end of file occurred. (Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
							line = line.strip()
							items = shlex.split(line)
							if len(items)==0: 
								line, sDict, count = BuildNameSub(g)#	Modified to parse index substitutions
								if not line: break
								lineCount += 1
								line = line.strip()
								#Parse the line into its space delimited components, recognising quoted items containing spaces
								items = shlex.split(line)
								if items[0]==":" and items[-1]==":=":
									# If AMPL has partitioned the columns
									#		:    20  21  22  23  24  25    :=
									#		1     0   0   0   0   0   0
									#		2     0   0   0   0   0   0
									cols = shlex.split(line)
									cols = cols[1:-1]
									for i in range(0,len(cols)):#	Index Substitutions
										if cols[i] in sDict:
											cols[i] = sDict[cols[i]]
								elif items[0]!=";":
									break
							if items[0]==";":
								NotFinished = False
								break
							for i in range(0,len(cols)):
								try:
									value = items[i+1]
								except:
									raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred in line "+repr(line)+" when converting the last item '"+repr(items[-1])+"' to a number.\n\n"
									+"\n\n(Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
								indices[stars[(0!=transpose)]]=items[0]
								indices[stars[(1!=transpose)]]=cols[i]
								# Convert any values into floats
								for j in range(len(indices)):
									try:
										indices[j]=float(indices[j])
									except ValueError:
										pass
								if len(indices)==1:
									index=indices[0]
								else:
									index=tuple(indices)
								if value != ".": # AJM 20130701 Skip missing items
									try:
										if dataItem[index]!=value:
											dataItem[index]=value
											changed=True
									except: continue
				else:
					while True:
						line = g.readline()
						lineCount += 1
						if not line: 
							raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an unexpected end of file occurred. (Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
						line = line.strip()
						items = shlex.split(line)
						if len(items)==0: continue
						if items[0]==";": break
						if len(items)<2:
							raise Exception("When writing new values for data item "+repr(name)+" to the sheet, the line "+repr(line)+" did not contain an index and a value.\n\n"
							+standardErrorHelp+"\n\n(Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
						try:
							value = float(items[-1])
						except:
							raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred in line "+repr(line)+" when converting the last item '"+repr(items[-1])+"' to a number.\n\n"
							+standardErrorHelp+"\n\n(Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")
						index = items[0:-1]
						# Convert any values into floats
						for j in range(len(index)):
							try:
								index[j]=float(index[j])
							except ValueError:
								pass
						# print tuple(index),value
						try:
							if len(index)==1:
								index2 = index[0]  # The index is a single value, not a tuple
							else:
								index2 = tuple(index)# The index is a tuple of two or more items
							if dataItem[index2]!=value:
								dataItem[index2]=value
								changed = True
						except:
							raise Exception("When writing new values for data item "+repr(name)+" to the sheet, an error occurred in line "+repr(line)+" when assigning the value "+repr(items[-1])+" to index "+repr(index2)+". (Line "+repr(lineCount)+" of file "+repr(resultsfilename)+")")            
				if changed:
					 itemsLoaded = itemsLoaded+name+"* "
					 itemsChanged = True
				else:
					itemsLoaded = itemsLoaded+name+" "
		finally:
			g.close()
		if itemsLoaded=="":
			print "## No results were loaded into the sheet."
		elif not itemsChanged:
			print "## Results loaded for data items:", itemsLoaded
		else :
			print "## Results loaded for data items:", itemsLoaded
			print "##   (*=data item values changed on sheet)"
	else:
		print "## "+language+" did not complete (Error Code %d); no solution is available. The sheet has not been changed." % exitCode
	print "## Done"

DoRun()